package de.uni_passau.fim.dagmar.util;

/**
 * Builder for XML documents.
 */
public class XmlBuilder {
    private static final String HEADER = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";

    private StringBuilder builder;

    private int identation;

    /**
     * Returns a wrapped version of the specified string where XML special
     * character are replaced by their escaped counterparts.
     * @param string the string to wrap.
     * @return a wrapped version of the specified string.
     */
    public static String escape(String string) {
        StringBuilder b = new StringBuilder();

        int length = string.length();

        for (int i = 0; i < length; i++) {
            char ch = string.charAt(i);

            switch (ch) {
            case '&':
                b.append("&amp;");
                break;
            case '"':
                b.append("&quot;");
                break;
            case '\'':
                b.append("&apos;");
                break;
            case '<':
                b.append("&lt;");
                break;
            case '>':
                b.append("&gt;");
                break;
            default:
                b.append(ch);
            }
        }

        return b.toString();
    }

    /**
     * Constructs a new {@code XmlBuilder}.
     */
    public XmlBuilder() {
        builder = new StringBuilder(HEADER);
        identation = 0;
    }

    /*
     * Appends the specified line to the built document.
     */
    private void appendLine(String line) {
        for (int i = 0; i < identation; i++) {
            builder.append(' ');
        }

        builder.append(line).append('\n');
    }

    /*
     * Creates a string representing the specified attribute values.
     */
    private String createAttributeString(String[] attributes) {
        if ((attributes.length & 1) == 1) {
            throw new IllegalArgumentException();
        }

        StringBuilder b = new StringBuilder();

        for (int i = 0; i < attributes.length; i += 2) {
            b.append(' ').append(attributes[i]).append("=\"")
                    .append(attributes[i + 1]).append('"');
        }

        return b.toString();
    }

    /**
     * Appends an opening tag to the built document.
     * @param tagName the name of the tag.
     * @param attributes the attributes and their values.
     */
    public void appendOpenTag(String tagName, String... attributes) {
        appendLine(new StringBuilder("<").append(tagName)
                .append(createAttributeString(attributes)).append('>')
                .toString());
        identation += 4;
    }

    /**
     * Appends a closing tag to the built document.
     * @param tagName the name of the tag.
     */
    public void appendCloseTag(String tagName) {
        identation -= 4;
        appendLine(new StringBuilder("</").append(tagName).append('>')
                .toString());
    }

    /**
     * Appends the specified XML content to the built document.
     * @param content the XML content.
     */
    public void appendContent(String content) {
        appendLine(content);
    }

    /**
     * Appends an opening and closing tag with the specified name enclosing
     * the specified content to the built document.
     * @param tagName the name of the tags.
     * @param content the content enclosed by the tags.
     * @param attributes the attributes and their values.
     */
    public void append(String tagName, String content, String... attributes) {
        appendLine(new StringBuilder("<").append(tagName)
                .append(createAttributeString(attributes)).append('>')
                .append(content).append("</").append(tagName).append('>')
                .toString());
    }

    /**
     * Appends a singleton tag to the built document.
     * @param tagName the name of the singleton tag.
     * @param attributes the attributes and their values.
     */
    public void appendSingletonTag(String tagName, String... attributes) {
        appendLine(new StringBuilder("<").append(tagName)
                .append(createAttributeString(attributes)).append(" />")
                .toString());
    }

    /**
     * Returns the built document as a {@code String}.
     */
    @Override
    public String toString() {
        return builder.toString();
    }
}
