package de.uni_passau.fim.dagmar.util;

/**
 * Union find structure.
 */
public class UnionFind {
    /*
     * The size of this structure. 
     */
    private int size;
    
    private int[] parent;
    private int[] rank;

    /**
     * Constructs a new union find structure with the specified size.
     * 
     * @param size the size of the structure.
     */
    public UnionFind(int size) {
        this.size = size;
        parent = new int[size];
        rank = new int[size];
        reset();
    }

    /**
     * Returns the canonical representative for the connected component
     * containing the specified element.
     * 
     * @param x
     *            the element the representative is returned for.
     * @return the canonical representative for the connected component
     *         containing the specified element.
     */
    public int find(final int x) {
        int p = parent[x];
        
        if (p != x) {
            p = find(p);
            parent[x] = p;
        }
        
        return p;
    }
    
    /**
     * Unifies the connected components of the specified elements.
     * 
     * @param x the first element
     * @param y the second element
     */
    public void union(int x, int y) {
        x = find(x);
        y = find(y);
        
        if (x != y) {
            if (rank[x] < rank[y]) {
                parent[x] = y;
            } else {
                parent[y] = x;
                
                if (rank[x] == rank[y]) {
                    rank[x]++;
                }
            }
        }
    }
    
    /**
     * Resets this structure. Each element is assigned to a singleton component.
     */
    public void reset() {
        for (int i = 0; i < size; i++) {
            parent[i] = i;
        }
    }

    /**
     * Returns if all elements are contained by a single connected component.
     * 
     * @return {@code true} if all elements are contained by a single connected
     *         component.
     */
    public boolean isConnected() {
        if (size > 0) {
            int value = find(parent[0]);
            for (int i = 1; i < size; i++) {
                if (find(parent[i]) != value) {
                    return false;
                }
            }
        }
        return true;
    }
}
