package de.uni_passau.fim.dagmar.util;

import java.util.HashMap;
import java.util.Map;

/**
 * Simple parser for arithmetic expressions. It supports the <code>+</code>,
 * <code>-</code>, <code>*</code>, {@code /}, <code>^</code>, and
 * <code>sqrt()</code> operations and arbitrary nestings of <code>()</code>
 * parentheses. Additionally, it supports the operators <code>|</code> and
 * {@code &}, which represent the maximum and minimum, respectively, and have
 * the least precedence. Values may be integers, floating points, and variables,
 * whose values are assigned by {@link #bindVariable(String, double)}.
 * Predefined variables are <code>n</code> and <code>nodeCount</code>, which are
 * initialized by the <code>nodeCount</code> parameter passed to the
 * constructor, <code>e</code>, <code>m</code>, and <code>edgeCount</code>,
 * which are initialized by the <code>edgeCount</code> parameter, and
 * <code>d</code> and <code>density</code> , which are initialized to
 * <code>density</code>.
 */
public class ArithmeticParser {
    private static class ParseException extends Exception {
        private static final long serialVersionUID = -487019427760403803L;
    }
    
    private Map<String, Double> variables;
    
    /**
     * Constructs a new instance and initializes the variable bindings
     * to the passed values.
     */
    public ArithmeticParser(int nodeCount, int edgeCount, double density) {
        variables = new HashMap<String, Double>();
        variables.put("nodeCount", (double) nodeCount);
        variables.put("n", (double) nodeCount);
        variables.put("edgeCount", (double) edgeCount);
        variables.put("m", (double) edgeCount);
        variables.put("e", (double) edgeCount);
        variables.put("density", (double) edgeCount);
        variables.put("d", density);
    }
    
    /**
     * Binds the variable specified by <code>variableName</code> to
     * <code>value</code>.
     */
    public void bindVariable(String variableName, double value) {
        variables.put(variableName, value);
    }
    
    /**
     * Parses the specified string.
     * 
     * @return the value of the arithmetic expression or <code>null</code>
     * if the string is ill-formated.
     */
    public Double parse(String string) {
        try {
            return internalParse(string);
        } catch (ParseException e) {
            return null;
        }
    }
    
    private double internalParse(String string) throws ParseException {
        string = trim(string);
        
        int pos = findBinarySymbol(string, "&|", false);
        
        if (pos == -1) {
            pos = findBinarySymbol(string, "+-", false);
        }
        
        if (pos == -1) {
            pos = findBinarySymbol(string, "*/", false);
        }
        
        if (pos == -1) {
            pos = findBinarySymbol(string, "^", true);
        }
        
        if (pos != -1) {
            return calculateBinary(internalParse(substring(string, 0, pos)),
                    internalParse(substring(string, pos + 1, string.length())),
                    string.charAt(pos));
        }
        
        if (string.startsWith("sqrt")) {
            return Math.sqrt(internalParse(string.substring(4)));
        } else if (string.startsWith("ceil")) {
            return Math.ceil(internalParse(string.substring(4)));
        } else if (string.startsWith("floor")) {
            return Math.floor(internalParse(string.substring(5)));
        } else if (string.startsWith("round")) {
            return Math.round(internalParse(string.substring(5)));
        }
        
        Double val = variables.get(string);
        if (val != null) {
            return val;
        }
        
        try {
            return Double.valueOf(string);
        } catch (NumberFormatException e) {
            throw new ParseException();
        }
    }
    
    private String substring(String string, int beginIndex, int endIndex) {
        if (beginIndex > endIndex) {
            return "";
        } else {
            return string.substring(beginIndex, endIndex);
        }
    }
    
    private String trim(String string) {
        string = string.trim();
        if (string.startsWith("(") && string.endsWith(")")) {
            int nesting = 1;
            for (int i = 1; i < string.length() - 1; i++) {
                char ch = string.charAt(i);
                if (ch == '(') {
                    nesting++;
                } else if (ch == ')') {
                    nesting--;
                } else if (nesting == 0) {
                    return string;
                }
            }
            
            return substring(string, 1, string.length() - 1);
        } else {
            return string;
        }
    }
    
    private int findBinarySymbol(String string, String symbols, boolean fromLeft) {
        int nesting = 0;
        for (int i = 0; i < string.length(); i++) {
            char ch = string.charAt(fromLeft ? i : string.length() - i - 1);
            if (fromLeft && ch == '(' || !fromLeft && ch == ')') {
                nesting++;
            } else if (fromLeft && ch == ')' || !fromLeft && ch == '(') {
                nesting--;
            } else if (nesting == 0) {
                if (i != 0 && symbols.indexOf(ch) != -1) {
                    return fromLeft? i : string.length() - i - 1;
                }
            }
        }
        return -1;
    }
    
    private double calculateBinary(double val1, double val2, char symbol) {
        switch (symbol) {
        case '|':
            return Math.max(val1, val2);
        case '&':
            return Math.min(val1, val2);
        case '+':
            return val1 + val2;
        case '-':
            return val1 - val2;
        case '*':
            return val1 * val2;
        case '/':
            return val1 / val2;
        case '^':
            return Math.pow(val1, val2);
        default:
            return 0;
        }
    }
}
