package de.uni_passau.fim.dagmar.arguments;

import java.math.BigDecimal;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * Utility class for handling program arguments.
 */
public class ProgramArguments {
    private Map<String, String> map;
    private List<String> freeStrings;
    
    protected ProgramArguments(String[] args) throws ProgramArgumentException {
        map = new HashMap<String, String>();
        freeStrings = new LinkedList<String>();
        
        String key = null;
        
        for (String arg : args) {
            if (arg.startsWith("-")) {
                key = arg.substring(1);
                map.put(key, null);
            } else if (key != null) {
                if (map.get(key) != null) {
                    throw new ProgramArgumentException("argument \"-" + key + "\" repeatedly specified");
                }
                
                map.put(key, arg);
                key = null;
            } else {
                freeStrings.add(arg);
            }
        }
    }
    
    protected String getString(String key) {
        return map.get(key);
    }
    
    protected Integer getInteger(String key) throws ProgramArgumentException {
        String string = map.get(key);
        
        if (string == null) {
            return null;
        } else try {
            return Integer.valueOf(string);
        } catch (NumberFormatException e) {
            throw new ProgramArgumentException("argument \"-" + key + "\" must be an integer");
        }
    }
    
    protected BigDecimal getBigDecimal(String key) throws ProgramArgumentException {
        String string = map.get(key);
        
        if (string == null) {
            return null;
        } else try {
            return new BigDecimal(string);
        } catch (NumberFormatException e) {
            throw new ProgramArgumentException("argument \"-" + key + "\" must be a number");
        }
    }
    
    protected Long getLong(String key) throws ProgramArgumentException {
        String string = map.get(key);
        
        if (string == null) {
            return null;
        } else try {
            return Long.valueOf(string);
        } catch (NumberFormatException e) {
            throw new ProgramArgumentException("argument \"-" + key + "\" must be a long integer");
        }
    }
    
    protected boolean containsFlag(String key) {
        return map.containsKey(key);
    }
    
    public List<String> getFreeStrings() {
        return freeStrings;
    }
}
