package de.uni_passau.fim.dagmar.arguments;

import java.io.File;
import java.math.BigDecimal;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import de.uni_passau.fim.dagmar.arguments.DAGmarProgramArguments.EmbeddingKind;
import de.uni_passau.fim.dagmar.util.Range;

public class MultiDAGmarProgramArguments extends SharedProgramArguments {
    private static final Pattern INTEGER_RANGE_PATTERN1 = Pattern
            .compile("([\\d]+)\\s+to\\s+([\\d]+)\\s+by\\s+([\\d]+)");
    private static final Pattern INTEGER_RANGE_PATTERN2 = Pattern
            .compile("([\\d]+)\\s+to\\s+([\\d]+)");
    private static final Pattern INTEGER_RANGE_PATTERN3 = Pattern
            .compile("([\\d]+)");
    private static final Pattern DECIMAL_RANGE_PATTERN1 = Pattern
            .compile("([\\d\\.]+)\\s+to\\s+([\\d\\.]+)\\s+by\\s+([\\d\\.]+)");
    private static final Pattern DECIMAL_RANGE_PATTERN2 = Pattern
            .compile("([\\d\\.]+)\\s+to\\s+([\\d\\.]+)");
    private static final Pattern DECIMAL_RANGE_PATTERN3 = Pattern
            .compile("([\\d\\.]+)");

    private List<Range<Integer>> nodeCounts;
    
    private List<Range<BigDecimal>> densities;
    
    private List<Range<Integer>> instances;
    
    private Long seed;
    private boolean connected;
    
    private String levelCountFormula;
    private String levelWidthFormula;
    private boolean leveled;
    private EmbeddingKind embeddingKind;
    
    private boolean flatDirectory;
    private String filePattern;
    private File targetDirectory;

    public MultiDAGmarProgramArguments(String[] args)
            throws ProgramArgumentException {
        super(args);
        
        nodeCounts = getIntegerRange("n");
        if (nodeCounts == null) {
            throw new ProgramArgumentException("node counts not specified"); 
        }
        
        densities = getBigDecimalRange("d");
        if (densities == null) {
            throw new ProgramArgumentException("densities not specified");
        }
        
        instances = getIntegerRange("i");
        
        seed = getLong("s");
        connected = containsFlag("c");
        
        String levelString = getString("l");
        if (levelString == null) {
            levelCountFormula = "n";
            levelWidthFormula = "1";
            leveled = false;
        } else if (levelString.equals("goldenratio")) {
            levelCountFormula = "ceil(sqrt(1.2 * 2 * n / (1 + sqrt(5))))";
            levelWidthFormula = "ceil((1 + sqrt(5)) / 2 * k)";
            leveled = true;
        } else {
            int pos = levelString.indexOf(',');
            if (0 < pos && pos < levelString.length() - 1) {
                levelCountFormula = levelString.substring(0, pos);
                levelWidthFormula = levelString.substring(pos + 1, levelString.length());
            } else {
                levelCountFormula = levelString;
                levelWidthFormula = "n";
            }
            leveled = true;
        }
        
        embeddingKind = obtainEmbedding();
        
        flatDirectory = containsFlag("flat");
        filePattern = getString("f");
        if (filePattern == null) {
            throw new ProgramArgumentException("file pattern not specified");
        }
        
        List<String> strings = getFreeStrings();
        if (!strings.isEmpty()) {
            targetDirectory = new File(strings.iterator().next());
        } else {
            throw new ProgramArgumentException("target directory not specified");
        }
        if (strings.size() > 1) {
            throw new ProgramArgumentException("multiple target directories specified");
        }
        
        if (!targetDirectory.exists()) {
            boolean success = targetDirectory.mkdirs();
            if (!success) {
                throw new ProgramArgumentException("specified target directory could not be created");
            }
        } else if (targetDirectory.isFile()) {
            throw new ProgramArgumentException("specified target directory is a regular file");
        }
    }
    
    private List<Range<BigDecimal>> getBigDecimalRange(String key)
            throws ProgramArgumentException {
        String string = getString(key);

        if (string == null)
            return null;

        List<Range<BigDecimal>> result = new LinkedList<Range<BigDecimal>>();
        for (String part : string.split(",")) {
            part = part.trim();

            Matcher m;

            if ((m = DECIMAL_RANGE_PATTERN1.matcher(part)).matches()) {
                result.add(new Range<BigDecimal>(new BigDecimal(m.group(1)),
                        new BigDecimal(m.group(2)), new BigDecimal(m.group(3))));
            } else if ((m = DECIMAL_RANGE_PATTERN2.matcher(part)).matches()) {
                result.add(new Range<BigDecimal>(new BigDecimal(m.group(1)),
                        new BigDecimal(m.group(2)), new BigDecimal(1)));
            } else if ((m = DECIMAL_RANGE_PATTERN3.matcher(part)).matches()) {
                result.add(new Range<BigDecimal>(new BigDecimal(m.group(1)),
                        new BigDecimal(m.group(1)), new BigDecimal(1)));
            } else {
                throw new ProgramArgumentException(
                        "illegal range format of \"-" + key + "\" argument");
            }
        }

        return result;
    }
    
    private List<Range<Integer>> getIntegerRange(String key)
            throws ProgramArgumentException {
        String string = getString(key);      

        List<Range<Integer>> result = new LinkedList<Range<Integer>>();
        
        if (string == null) {
            result.add(new Range<Integer>(0, 0, 1));
            return result;
        }

        for (String part : string.split(",")) {
            part = part.trim();

            Matcher m;

            if ((m = INTEGER_RANGE_PATTERN1.matcher(part)).matches()) {
                result.add(new Range<Integer>(Integer.valueOf(m.group(1)),
                        Integer.valueOf(m.group(2)), Integer.valueOf(m.group(3))));
            } else if ((m = INTEGER_RANGE_PATTERN2.matcher(part)).matches()) {
                result.add(new Range<Integer>(Integer.valueOf(m.group(1)),
                        Integer.valueOf(m.group(2)), Integer.valueOf(1)));
            } else if ((m = INTEGER_RANGE_PATTERN3.matcher(part)).matches()) {
                result.add(new Range<Integer>(Integer.valueOf(m.group(1)),
                        Integer.valueOf(m.group(1)), Integer.valueOf(1)));
            } else {
                throw new ProgramArgumentException(
                        "illegal range format of \"-" + key + "\" argument");
            }
        }

        return result;
    }
    
    private EmbeddingKind obtainEmbedding() throws ProgramArgumentException {
        EnumSet<EmbeddingKind> embeddings = EnumSet.noneOf(EmbeddingKind.class);
        
        if (containsFlag("p")) {
            embeddings.add(EmbeddingKind.PROPER);
        }
        
        if (containsFlag("el")) {
            embeddings.add(EmbeddingKind.EMBEDDING);
        }
        
        if (containsFlag("ed")) {
            embeddings.add(EmbeddingKind.DUMMY_EMBEDDING);
        }
         
        if (embeddings.size() > 1) {
            throw new ProgramArgumentException("options -p, -el, and -ed are mutually exclusive");
        }
        
        Iterator<EmbeddingKind> iter = embeddings.iterator();
        
        if (iter.hasNext()) {
            if (!leveled) {
                throw new ProgramArgumentException("options -p, -el, and -ed must be combined with -l");
            }
            return iter.next();
        } else {
            return EmbeddingKind.NONE;
        }
    }
    
    public List<Range<Integer>> getNodeCounts() {
        return nodeCounts;
    }
    
    public List<Range<BigDecimal>> getDensities() {
        return densities;
    }
    
    public List<Range<Integer>> getInstances() {
        return instances;
    }
    
    public Long getSeed() {
        return seed;
    }
    
    public boolean isConnected() {
        return connected;
    }
    
    public EmbeddingKind getEmbeddingKind() {
        return embeddingKind;
    }
    
    public String getLevelCountFormula() {
        return levelCountFormula;
    }
    
    public String getLevelWidthFormula() {
        return levelWidthFormula;
    }
    
    public boolean isLeveled() {
        return leveled;
    }
    
    public boolean isFlatDirectory() {
        return flatDirectory;
    }
    
    public File getTargetDirectory() {
        return targetDirectory;
    }
    
    public String getFilePattern() {
        return filePattern;
    }
}