package de.uni_passau.fim.dagmar.arguments;

import java.io.File;
import java.math.BigDecimal;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class DAGmarProgramArguments extends SharedProgramArguments {
    private static final Pattern LEVEL_PATTERN = Pattern.compile("(\\d+)(?:\\,(\\d+))?");
    
    public enum EmbeddingKind {
        NONE,
        PROPER,
        EMBEDDING,
        DUMMY_EMBEDDING
    }
    
    private Integer nodeCount;
    private Integer edgeCount;
    private BigDecimal density;
    private Integer levelCount;
    private Integer levelWidth;
    private EmbeddingKind embeddingKind;
    private Long seed;
    private boolean connected;
    private Integer instance;
    private boolean leveled;
    private String filePattern;
    
    public DAGmarProgramArguments(String[] args) throws ProgramArgumentException {
        super(args);
        
        nodeCount = getInteger("n");
        edgeCount = getInteger("e");
        density = getBigDecimal("d");
        
        String levelString = getString("l");
        if (levelString != null) {
            Matcher m = LEVEL_PATTERN.matcher(levelString);
            if (m.matches()) {
                levelCount = Integer.valueOf(m.group(1));
                String group2 = m.group(2);
                levelWidth = group2 == null ? nodeCount : Integer.valueOf(group2);
            }
        }
        
        seed = getLong("s");
        connected = containsFlag("c");
        instance = getInteger("i");
        
        filePattern = getString("f");
        
        if (nodeCount == null) {
            throw new ProgramArgumentException("node count not specified");
        }

        if (!((edgeCount == null) ^ (density == null))) {
            throw new ProgramArgumentException(
                "exactly one of edge count and density must be specified");
        }

        if (edgeCount == null) {
            edgeCount = density.multiply(new BigDecimal(nodeCount)).intValue();
        }

        if (levelCount == null) {
            leveled = false;
            levelCount = nodeCount;
            levelWidth = 1;
        } else {
            leveled = true;
        }

        if (levelWidth == null) {
            levelWidth = nodeCount;
        } else {
            levelWidth = Math.min(levelWidth, nodeCount);
        }

        if (instance == null) {
            instance = 0;
        }
        
        embeddingKind = obtainEmbedding(leveled);

        if (levelWidth * levelCount < nodeCount) {
            throw new ProgramArgumentException(
                "not enough space to place all nodes");
        }

        if (levelCount <= 1 && edgeCount > 0) {
            throw new ProgramArgumentException(
                "graph on a single level cannot have any edges");            
        }
        
        if (connected && levelCount <= 1 && nodeCount > 1) {
            throw new ProgramArgumentException(
                "graph on a single level cannot be connected");
        }
    }
    
    public int getNodeCount() {
        return nodeCount;
    }
    
    public int getEdgeCount() {
        return edgeCount;
    }
    
    public EmbeddingKind getEmbeddingKind() {
        return embeddingKind;
    }
    
    public boolean isConnected() {
        return connected;
    }
    
    public boolean isLeveled() {
        return leveled;
    }
    
    public int getLevelCount() {
        return levelCount;
    }
    
    public int getLevelWidth() {
        return levelWidth;
    }
    
    public int getInstance() {
        return instance;
    }
    
    public Long getSeed() {
        return seed;
    }
    
    public File getFile() {
        if (filePattern == null) return null;
        
        return new File(filePattern + ".graphml");
    }
}
