package de.uni_passau.fim.dagmar;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.math.BigDecimal;
import java.util.Random;

import de.uni_passau.fim.dagmar.arguments.MultiDAGmarProgramArguments;
import de.uni_passau.fim.dagmar.arguments.ProgramArgumentException;
import de.uni_passau.fim.dagmar.arguments.DAGmarProgramArguments.EmbeddingKind;
import de.uni_passau.fim.dagmar.util.ArithmeticParser;
import de.uni_passau.fim.dagmar.util.Range;

/**
 * Program for generating multiple graphs with {@link DAGmar}. The graphs
 * are saved to the directory specified by the program arguments.
 */
public class MultiDAGmar {
    
    //
    // range: "10" | "10 to 100" | "10 to 100 by 10" | "<range>, <range>"
    //
    private static final String USAGE = "Usage: java -jar dagmar.jar multi "
        + "-n <node count ranges> -d <density ranges> [ -i <instance ranges> ] [ -c ] "
        + "[ -l <level count/formula>[,<level size/formula>] [ -p | -el | -ed ] ] "
        + "[ -s <seed> ] [ -flat ] "
        + "[ -f <file prefix> ] <target directory>\n";

    /**
     * Program entry point.
     * 
     * @param args the program arguments.
     * @throws IOException is thrown if a graph could not be saved.
     */
    public static void main(String[] args) throws IOException {
        if (args.length == 0) {
            System.err.println(USAGE);
            return;
        }

        try {
            MultiDAGmarProgramArguments arguments
                = new MultiDAGmarProgramArguments(args);
            Long seed = arguments.getSeed();
            Random masterRandom
                = (seed == null) ? new Random() : new Random(seed);
            forEachDensity(arguments, masterRandom);
        } catch (ProgramArgumentException e) {
            System.err.println(e.getMessage());
        }
    }

    private static void forEachDensity(MultiDAGmarProgramArguments arguments,
            Random masterRandom) throws ProgramArgumentException, IOException {
        String path = arguments.getTargetDirectory().getCanonicalPath();

        for (Range<BigDecimal> range : arguments.getDensities()) {
            for (BigDecimal density = range.getFrom(); density.compareTo(range
                    .getTo()) <= 0; density = density.add(range.getStep())) {
                String prefix = path + "/";
                if (!arguments.isFlatDirectory()) {
                    prefix += density.toPlainString() + "/";

                    File dir = new File(prefix);
                    if (!dir.exists()) {
                        boolean success = dir.mkdir();
                        if (!success) {
                            throw new ProgramArgumentException("directory \""
                                    + dir + "\" could not be created");
                        }
                    } else if (!dir.isDirectory()) {
                        throw new ProgramArgumentException("path \"" + dir
                                + "\" is not a directory");
                    }
                }

                Random random = new Random(masterRandom.nextLong());
                forEachNodeCount(arguments, density, prefix, random);
            }
        }
    }

    private static void forEachNodeCount(MultiDAGmarProgramArguments arguments,
            BigDecimal density, String prefix, Random random)
            throws IOException, ProgramArgumentException {
        for (Range<Integer> range : arguments.getNodeCounts()) {
            for (int nodeCount = range.getFrom();
                    nodeCount <= range.getTo();
                    nodeCount += range.getStep()) {
                int edgeCount = density.multiply(new BigDecimal(nodeCount))
                        .intValue();

                if ((edgeCount > nodeCount * (nodeCount - 1) / 2)
                        || (arguments.isConnected() && edgeCount < nodeCount - 1)) {
                    System.err.println("Warning: no connected graph for "
                            + nodeCount + " nodes and " + edgeCount + " edges");
                    continue;
                }

                ArithmeticParser parser = new ArithmeticParser(nodeCount,
                        edgeCount, density.doubleValue());

                Double parsedLevelCount = parser.parse(arguments
                        .getLevelCountFormula());
                if (parsedLevelCount == null) {
                    throw new ProgramArgumentException("illegal formula \""
                            + arguments.getLevelCountFormula() + "\"");
                }
                int levelCount = (int) Math.round(parsedLevelCount);

                parser.bindVariable("levelCount", levelCount);
                parser.bindVariable("k", levelCount);

                Double parsedLevelWidth = parser.parse(arguments
                        .getLevelWidthFormula());
                if (parsedLevelWidth == null) {
                    throw new ProgramArgumentException("illegal formula \""
                            + arguments.getLevelWidthFormula() + "\"");
                }
                int levelWidth = (int) Math.round(parsedLevelWidth);

                boolean proper
                    = arguments.getEmbeddingKind() == EmbeddingKind.PROPER; 
                int maxEdges = DAGmar.countMaxEdges(nodeCount, levelCount,
                    levelWidth, proper);
                if (edgeCount > maxEdges) {
                    System.err.println("Warning: " + levelCount
                            + " levels with at most " + levelWidth
                            + " nodes each do not allow for "
                            + (proper ? "proper " : "") + "graphs with "
                            + nodeCount + " nodes and " + edgeCount + " edges");
                    continue;
                }

                forEachInstance(arguments, nodeCount, edgeCount, levelCount,
                        levelWidth, prefix, random);
            }
        }
    }

    private static void forEachInstance(MultiDAGmarProgramArguments arguments,
            int nodeCount, int edgeCount, int levelCount, int levelWidth,
            String prefix, Random random) throws IOException {
        Long generalSeed = random.nextLong();
        for (Range<Integer> range : arguments.getInstances()) {
            for (int instance = range.getFrom();
                    instance <= range.getTo();
                    instance += range.getStep()) {
                String fileName = prefix + arguments.getFilePattern() + "_n"
                    + nodeCount + "_e" + edgeCount + "_i" + instance + ".graphml";

                Random instanceRandom = new Random(generalSeed);
                for (int i = 0; i < instance; i++) {
                    instanceRandom.nextLong();
                }

                PrintStream out = new PrintStream(
                        new FileOutputStream(fileName));
                
                EmbeddingKind embeddingKind = arguments.getEmbeddingKind();
                boolean isEmbedded = embeddingKind == EmbeddingKind.EMBEDDING
                        || embeddingKind == EmbeddingKind.DUMMY_EMBEDDING;
                boolean hasDummies = embeddingKind == EmbeddingKind.DUMMY_EMBEDDING;

                DAGmar generator = new DAGmar();
                EmbeddedLevelGraph graph = generator.generate(nodeCount, edgeCount,
                        arguments.isConnected(), arguments.getEmbeddingKind(),
                        levelCount, levelWidth, instanceRandom.nextLong());
                out.print(new GraphMLBuilder(arguments.isLeveled(), isEmbedded,
                        hasDummies, graph, "G"));

                out.close();
            }
        }
    }
}
