package de.uni_passau.fim.dagmar;

import static de.uni_passau.fim.dagmar.util.Combinatorics.binom;
import static java.lang.Math.max;

import java.math.BigInteger;
import java.util.Random;

import de.uni_passau.fim.dagmar.util.CollectionUtil;
import de.uni_passau.fim.dagmar.util.Combinatorics;

/**
 * Generator for levelings.
 */
public class LevelingGenerator {
    private Random random;
    
    /**
     * Constructs the generator for the specified source of randomness.
     * 
     * @param random the source of randomness.
     */
    public LevelingGenerator(Random random) {
        this.random = random;
    }
    
    /**
     * Generates a leveling for the specified graph with the specified
     * parameters.
     * 
     * @param graph
     *            the graph for which the leveling to generate.
     * @param edgeCount
     *            the number of potential edges the leveling has to support.
     * @param levelWidth
     *            the maximum number of vertices per level.
     * @param proper
     *            determines if edges shall be short.
     * @param consecutive
     *            determines if non-empty levels may be interleaved by empty
     *            ones.
     * @return {@code true} if there exists at least one leveling for the
     *         specified parameters.
     */
    public boolean generateLeveling(EmbeddedLevelGraph graph, int edgeCount,
            int levelWidth, boolean proper, boolean consecutive) {
        int nodeCount = graph.getNodeCount();
        int maxLevelCount = graph.getMaxLevelCount();
        
        Enumerator enumerator = new Enumerator(nodeCount, edgeCount, levelWidth,
            graph.getMaxLevelCount(), proper, consecutive);
        
        int[] nodes = new int[nodeCount];
        for (int i = 0; i < nodeCount; i++) {
            nodes[i] = i;
        }
        
        int remainingEdgeCount = edgeCount;
        
        // n is the remaining number of vertices to place
        int n = nodeCount;
        
        // nAbove is the number of vertices placed one level above the current
        int nAbove = 0;
        
        for (int level = 0; n > 0; level++) {
            int visiblePredecessors = proper ? nAbove : nodeCount - n;
            
            Combinatorics combinatorics = new Combinatorics();
            
            int minCount = consecutive && n < nodeCount ? 1 : 0;
            int maxCount = Math.min(n, levelWidth);
            for (int count = minCount; count <= maxCount; count++) {
                int newEdgePlaces = visiblePredecessors * count;
                int newRemainingEdgeCount
                    = max(0, remainingEdgeCount - newEdgePlaces);

                // In sum O(1).
                BigInteger weight = binom(n, count).multiply(
                        enumerator.enumerate(n - count,
                                             maxLevelCount - level - 1,
                                             newRemainingEdgeCount,
                                             count));
                
                combinatorics.add(count, weight);
            }
            
            
            if (combinatorics.isEmpty()) return false;
            
            int actualCount = combinatorics.draw(random);
            for (int i = 0; i < actualCount; i++) {
                int r = random.nextInt(n);
                graph.setNodeLevel(nodes[r], level);
                CollectionUtil.swap(nodes, r, n - 1);
                n--;
            }
            
            nAbove = actualCount;
            int actualNewEdgePlaces = visiblePredecessors * actualCount;
            remainingEdgeCount = max(0, remainingEdgeCount - actualNewEdgePlaces);
        }
        
        return true;
    }
}
