package de.uni_passau.fim.dagmar;

import de.uni_passau.fim.dagmar.util.XmlBuilder;

/**
 * Builder for GraphML files. If it is constructed by
 * {@link #GraphMLBuilder(boolean, boolean, boolean)}, then multiple graphs can
 * be included in a single GraphML file, but the builder has to be explicitly
 * closed by {@link #close()} before calling {@link #toString()}.
 */
public class GraphMLBuilder {
    private static final String TAG_GRAPH_ML = "graphml";
    private static final String NAME_XMLNS = "xmlns";
    private static final String CONTENT_XMLNS = "http://graphml.graphdrawing.org/xmlns";
    private static final String NAME_XMLNS_XSI = "xmlns:xsi";
    private static final String CONTENT_XMLNS_XSI = "http://www.w3.org/2001/XMLSchema-instance";
    private static final String NAME_XSI_SCHEMA_LOCATION = "xsi:schemaLocation";
    private static final String CONTENT_XSI_SCHEMA_LOCATION = "http://graphml.graphdrawing.org/xmlns http://graphml.graphdrawing.org/xmlns/1.0/graphml.xsd";

    private static final String TAG_GRAPH = "graph";
    private static final String NAME_EDGE_DEFAULT = "edgedefault";
    private static final String CONTENT_EDGE_DEFAULT = "directed";

    private static final String TAG_KEY = "key";
    private static final String NAME_ID = "id";
    private static final String NAME_FOR = "for";
    private static final String NAME_ATTR_NAME = "attr.name";
    private static final String NAME_ATTR_TYPE = "attr.type";

    private static final String CONTENT_KEY_LEVEL_FOR = "node";
    private static final String CONTENT_KEY_LEVEL_ID = "level";
    private static final String CONTENT_KEY_LEVEL_ATTR_NAME = "hierarchy.level";
    private static final String CONTENT_KEY_LEVEL_ATTR_TYPE = "int";
    
    private static final String CONTENT_KEY_POS_FOR = "node";
    private static final String CONTENT_KEY_POS_ID = "pos";
    private static final String CONTENT_KEY_POS_ATTR_NAME = "hierarchy.pos";
    private static final String CONTENT_KEY_POS_ATTR_TYPE = "int";
    
    private static final String CONTENT_KEY_DUMMY_FOR = "node";
    private static final String CONTENT_KEY_DUMMY_ID = "dummy";
    private static final String CONTENT_KEY_DUMMY_ATTR_NAME = "dummy";
    private static final String CONTENT_KEY_DUMMY_ATTR_TYPE = "boolean";

    private static final String TAG_NODE = "node";

    private static final String TAG_DATA = "data";
    private static final String NAME_KEY = "key";

    private static final String TAG_EDGE = "edge";
    private static final String NAME_SOURCE = "source";
    private static final String NAME_TARGET = "target";

    private XmlBuilder builder;

    private boolean isLeveled;
    private boolean isEmbedded;
    private boolean hasDummies;
    
    private boolean closed;

    /**
     * Constructs a new builder for GraphML files.
     * 
     * @param isLeveled
     *            denotes if the builder includes level attributes.
     * @param isEmbedded
     *            denotes if the builder includes attributes for the position of
     *            the nodes on their level.
     * @param hasDummies
     *            denotes if the builder includes attributes that designate some
     *            nodes as dummies.
     */
    public GraphMLBuilder(boolean isLeveled, boolean isEmbedded,
            boolean hasDummies) {
        this.isLeveled = isLeveled;
        builder = new XmlBuilder();
        builder.appendOpenTag(TAG_GRAPH_ML, NAME_XMLNS, CONTENT_XMLNS,
                NAME_XMLNS_XSI, CONTENT_XMLNS_XSI, NAME_XSI_SCHEMA_LOCATION,
                CONTENT_XSI_SCHEMA_LOCATION);
        if (isLeveled) {
            builder.appendSingletonTag(TAG_KEY, NAME_ID, CONTENT_KEY_LEVEL_ID,
                    NAME_FOR, CONTENT_KEY_LEVEL_FOR, NAME_ATTR_NAME,
                    CONTENT_KEY_LEVEL_ATTR_NAME, NAME_ATTR_TYPE,
                    CONTENT_KEY_LEVEL_ATTR_TYPE);
        }
        if (isEmbedded) {
            builder.appendSingletonTag(TAG_KEY, NAME_ID, CONTENT_KEY_POS_ID,
                    NAME_FOR, CONTENT_KEY_POS_FOR, NAME_ATTR_NAME,
                    CONTENT_KEY_POS_ATTR_NAME, NAME_ATTR_TYPE,
                    CONTENT_KEY_POS_ATTR_TYPE);
        }
        if (hasDummies) {
            builder.appendSingletonTag(TAG_KEY, NAME_ID, CONTENT_KEY_DUMMY_ID,
                    NAME_FOR, CONTENT_KEY_DUMMY_FOR, NAME_ATTR_NAME,
                    CONTENT_KEY_DUMMY_ATTR_NAME, NAME_ATTR_TYPE,
                    CONTENT_KEY_DUMMY_ATTR_TYPE);
        }
    }

    /**
     * Constructs a new builder for GraphML files. The builder is automatically
     * appended the specified graph and closed.
     * 
     * @param isLeveled
     *            denotes if the builder includes level attributes.
     * @param isEmbedded
     *            denotes if the builder includes attributes for the position of
     *            the nodes on their level.
     * @param hasDummies
     *            denotes if the builder includes attributes that designate some
     *            nodes as dummies.
     */
    public GraphMLBuilder(boolean isLeveled, boolean isEmbedded,
            boolean hasDummies, EmbeddedLevelGraph graph, String id) {
        this(isLeveled, isEmbedded, hasDummies);
        append(graph, id);
        close();
    }

    /**
     * Appends the specified graph to this builder.
     * 
     * @param graph
     *            the graph to append.
     * @param id
     *            the id attribute of the specified graph in the GraphML file.
     * @throws IllegalStateException
     *             if this builder has already been closed, either by a call to
     *             {@link #close()}, or as it has been constructed by
     *             {@link #GraphMLBuilder(boolean, boolean, boolean)}.
     */
    public void append(EmbeddedLevelGraph graph, String id) {
        if (closed) {
            throw new IllegalStateException(
                    "The builder has already been closed");
        }
        
        int nodeCount = graph.getNodeCount();

        builder.appendOpenTag(TAG_GRAPH, NAME_EDGE_DEFAULT,
                CONTENT_EDGE_DEFAULT);

        for (int i = 0; i < nodeCount; i++) {
            if (isLeveled) {
                builder.appendOpenTag(TAG_NODE, NAME_ID, "n" + i);
                builder.append(TAG_DATA, "" + graph.getNodeLevel(i), NAME_KEY,
                        CONTENT_KEY_LEVEL_ID);
                if (isEmbedded) {
                    builder.append(TAG_DATA, "" + graph.getNodePos(i), NAME_KEY,
                            CONTENT_KEY_POS_ID);
                }
                if (hasDummies) {
                    builder.append(TAG_DATA, "" + graph.isNodeDummy(i), NAME_KEY,
                            CONTENT_KEY_DUMMY_ID);
                }
                builder.appendCloseTag(TAG_NODE);
            } else {
                builder.appendSingletonTag(TAG_NODE, NAME_ID, "n" + i);
            }
        }

        for (int i = 0; i < nodeCount; i++) {
            for (int j : graph.getNeighbors(i)) {
                if (graph.getNodeLevel(i) < graph.getNodeLevel(j)) {
                    builder.appendSingletonTag(TAG_EDGE, NAME_SOURCE, "n" + i,
                            NAME_TARGET, "n" + j);
                }
            }
        }

        builder.appendCloseTag(TAG_GRAPH);
    }

    /**
     * Closes this builder to indicate that no more graphs are appended to this
     * builder.
     * 
     * @throws IllegalStateException
     *             if this builder has already been closed, either by a call to
     *             {@link #close()}, or as it has been constructed by
     *             {@link #GraphMLBuilder(boolean, boolean, boolean)}.
     */
    public void close() {
        if (closed) {
            throw new IllegalStateException(
                    "The builder has already been closed.");
        }
        
        builder.appendCloseTag(TAG_GRAPH_ML);
        closed = true;
    }

    /**
     * Returns a string representing the appended graphs in the GraphML file
     * format.
     * 
     * @throws IllegalStateException
     *             if this builder has not been closed yet.
     */
    public String toString() {
        if (!closed) {
            throw new IllegalStateException(
                    "The builder has not been closed yet.");
        }
        return builder.toString();
    }
}
