package de.uni_passau.fim.dagmar;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;

/**
 * A simple undirected graph based on adjacency lists. The nodes are represented
 * by the integers <code>0, ..., nodeCount - 1</code>.
 */
public class Graph {
    protected int nodeCount;
    
    private int edgeCount;

    // neighbors.get(i) are the neighbors of node i.
    protected ArrayList<Collection<Integer>> neighbors;

    /**
     * Constructs a new graph with the specified number of nodes.
     * 
     * @param nodeCount the number of nodes.
     */
    public Graph(int nodeCount) {
        if (nodeCount < 0) {
            throw new IllegalArgumentException("Illegal node count");
        }
        this.nodeCount = nodeCount;

        neighbors = new ArrayList<Collection<Integer>>();

        for (int i = 0; i < nodeCount; i++) {
            neighbors.add(new LinkedHashSet<Integer>());
        }

        edgeCount = 0;
    }

    /**
     * Adds an undirected edge between the specified source and target.
     * 
     * @param source the first node incident to the added edge
     * @param target the second node incident to the added edge
     */
    public void addEdge(int source, int target) {
        if (source >= nodeCount || target >= nodeCount) {
            throw new IllegalArgumentException("Illegal node number");
        }

        neighbors.get(source).add(target);
        neighbors.get(target).add(source);
        
        edgeCount++;
    }

    /**
     * Returns a collection of nodes adjacent to the specified node.
     * 
     * @param node the node
     * @return a collection of nodes adjacent to the specified node.
     */
    public Collection<Integer> getNeighbors(int node) {
        if (node < 0 || node >= nodeCount) {
            throw new IllegalArgumentException("Illegal node number");
        }

        return Collections.unmodifiableCollection(neighbors.get(node));
    }

    /**
     * Returns the number of nodes.
     * 
     * @return the number of nodes.
     */
    public int getNodeCount() {
        return nodeCount;
    }

    /**
     * Returns the number of edges.
     * 
     * @return the number of edges.
     */
    public int getEdgeCount() {
        return edgeCount;
    }

    /**
     * Returns if this graph contains the specified edge.
     * 
     * @param source the first node incident to the edge
     * @param target the second node incident to the edge
     * @return
     */
    public boolean containsEdge(int source, int target) {
        return neighbors.get(source).contains(target);
    }
}
