package de.uni_passau.fim.dagmar;

import static de.uni_passau.fim.dagmar.util.Combinatorics.binom;
import static java.lang.Math.max;
import static java.lang.Math.min;

import java.math.BigInteger;

import de.uni_passau.fim.dagmar.util.BigIntTable;

final class Enumerator {
    private int nodeCount;
    private int levelWidth;
    private boolean proper;
    private boolean consecutive;
    
    private BigIntTable table;
    
    /**
     * Constructs the look-up table. 
     * 
     * @param nodeCount
     * @param edgeCount
     * @param levelWidth
     * @param levelCount
     * @param proper
     *            is true if edges will be allowed only to join vertices of
     *            consecutive levels.
     * @param consecutive
     *            disallows levelings with non-empty levels interrupted by empty
     *            ones.
     */
    public Enumerator(int nodeCount, int edgeCount, int levelWidth, int levelCount, boolean proper, boolean consecutive) {
        this.nodeCount = nodeCount;
        this.levelWidth = levelWidth;
        this.proper = proper;
        this.consecutive = consecutive;
        if (proper) {
            // How many (proper) levelings are there with
            //     - n vertices to place
            //     - on at most k levels
            //     - supporting for at least edgeCount edges
            //     - if n0 vertices where placed on the level above?
            // n k edgeCount n0
            table = new BigIntTable(nodeCount + 1, levelCount + 1, edgeCount + 1, nodeCount + 1);
        } else {
            // How many (non-proper) levelings are there with
            //     - n vertices to place
            //     - on at most k levels
            //     - supporting for at least edgeCount edges?
            // n k edgeCount
            table = new BigIntTable(nodeCount + 1, levelCount + 1, edgeCount + 1);
        }
    }
    
    /**
     * Returns the number of levelings for the specified parameters. Results are
     * cached in this lookup-up table.
     * 
     * @param n
     *            number of remaining vertices.
     * @param k
     *            number of remaining levels. k > 0.
     * @param remainingEdgeCount
     *            remaining required number of places for potential edges > 0.
     * @param nAbove
     *            number of vertices in the level above
     * @return the number of levelings abiding the specified parameters.
     */
    public BigInteger enumerate(int n, int k, int remainingEdgeCount, int nAbove) {
        // The number of vertices in the levels above that could potentially
        // be joined with the vertices on the currently topmost level.
        // In the proper case these are exactly the vertices placed directly
        // above. In the non-proper case they comprise all yet placed vertices.
        int visiblePredecessors = proper ? nAbove : nodeCount - n;
        
        if (n < 0) {
            // Too many vertices were placed on higher levels. 
            return BigInteger.ZERO;
        } else if (n == 0) {
            // All vertices were placed on higher levels so there are none left
            // to be placed on the following levels. This defines a single
            // leveling, which is valid if there is enough space for the
            // required number of edges.
            if (remainingEdgeCount == 0) {
                return BigInteger.ONE;
            } else {
                return BigInteger.ZERO;
            }
        } else if (k == 0) {
            return BigInteger.ZERO;
        } else if (k == 1) {
            // We are on the last level so we must place all remaining vertices.
            // Check if the level is big enough and if there finally is enough
            // space for the required number of edges.
            int newEdgePlaces = visiblePredecessors * n;
            if (n <= levelWidth && remainingEdgeCount <= newEdgePlaces) {
                return BigInteger.ONE;
            } else {
                return BigInteger.ZERO;
            }
        }
        
        BigInteger cachedNumber;
        if (proper) {
            cachedNumber = table.get(n, k, remainingEdgeCount, nAbove);
        } else {
            // nAbove is ignored in the non-proper case so spare the space
            // required by caching a fourth parameter.
            cachedNumber = table.get(n, k, remainingEdgeCount);
        }
        if (cachedNumber != null) {
            return cachedNumber;
        }
        
        // If the leveling must be consecutive, i.e., there must be no empty
        // levels between nonempty levels, then this level must contain at least
        // one vertex if some vertices have been placed yet (n < nodeCount).
        int minCount = consecutive && n < nodeCount ? 1 : 0;
        int maxCount = min(n, levelWidth);
        
        BigInteger z = BigInteger.ZERO;
        for (int count = minCount; count <= maxCount; count++) {
            // count is the number of vertices placed on the currently topmost level.
            
            // each of the new vertices could potentially be joined with each
            // of the visible predecessors.
            int newEdgePlaces = visiblePredecessors * count;
            
            // a smaller number of edge places are yet to be create, but
            // no less than zero
            int newRemainingEdgeCount
                = max(0, remainingEdgeCount - newEdgePlaces);
            
            // Choose count of the n vertices to place on the currently topmost
            // level and find out how many possibilities there are to place the
            // rest.
            z = z.add(binom(n, count).multiply(
                    enumerate(n - count,
                              k - 1,
                              newRemainingEdgeCount,
                              count)));
        }
        
        // Storing the result in the table avoids redundant calculations.
        if (proper) {
            table.put(z, n, k, remainingEdgeCount, nAbove);
        } else {
            table.put(z, n, k, remainingEdgeCount);
        }
        
        return z;
    }
}
