package de.uni_passau.fim.dagmar;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Random;

/**
 * A simple undirected leveled graph. The nodes may optionally be assigned
 * positions, which determine the relative order of the nodes on a level.
 * A node can also be marked as a dummy. 
 */
public class EmbeddedLevelGraph extends Graph {

    private int levelCount;

    // levelByNode.get(i) is the level of node i.
    // A value of -1 means that the node has not been assigned to any levels yet.
    private ArrayList<Integer> levelByNode;
    
    // dummyByNode.get(i) if node i is considered a dummy.
    private ArrayList<Boolean> dummyByNode;
    
    // posByNode[i] is the position of node i on its level.
    private int[] posByNode;

    // levels[i] is the list of nodes on level i.
    protected final List<Integer>[] levels;

    /**
     * Constructs a new graph with the specified number of nodes and the
     * specified maximum number of levels.
     * 
     * @param nodeCount
     *            the number of nodes.
     * @param levelCount
     *            the maximum number of levels, i.e., a node can be placed on
     *            one of the levels <code>0</code> to
     *            <code>levelCount - 1</code>.
     */
    public EmbeddedLevelGraph(int nodeCount, int levelCount) {
        super(nodeCount);

        if (levelCount < 0) {
            throw new IllegalArgumentException("Illegal level count");
        }

        this.levelCount = levelCount;

        levelByNode = new ArrayList<Integer>(Collections.nCopies(nodeCount, -1));
        
        dummyByNode = new ArrayList<Boolean>(Collections.nCopies(nodeCount, false));

        @SuppressWarnings("unchecked")
        List<Integer>[] levels = (List<Integer>[]) new List<?>[levelCount];
        this.levels = levels;

        for (int level = 0; level < levelCount; level++) {
            levels[level] = new ArrayList<Integer>();
        }
    }

    /**
     * Places the specified node on the specified level.
     * 
     * @param node the node to place.
     * @param level the level the node is placed on.
     */
    public void setNodeLevel(int node, int level) {
        if (node < 0 || node >= nodeCount) {
            throw new IllegalArgumentException("Illegal node number");
        } else if (level < -1 || level >= levelCount) {
            throw new IllegalArgumentException("Illegal level number");
        }

        int oldLevel = levelByNode.get(node);

        if (oldLevel != -1) {
            levels[oldLevel].remove((Integer) node);
        }

        if (level != -1) {
            levels[level].add(node);
        }

        levelByNode.set(node, level);
    }

    /**
     * Returns the level of the specified node.
     * 
     * @param node the node whose level is returned.
     * @return the level of the specified node or -1 if the node has not been
     * assigned to any level yet.
     */
    public int getNodeLevel(int node) {
        return levelByNode.get(node);
    }
    
    /**
     * Returns the position of the specified node on its level. The positions
     * have to be assigned before this method is called the first time.
     * 
     * @param node the node whose position is returned.
     * @return the position of the specified node on its level.
     * @throws NullPointerException if the positions have not been assigned yet.
     */
    public int getNodePos(int node) {
        return posByNode[node];
    }
    
    /**
     * Returns if the specified node is a dummy.
     * 
     * @param node the node.
     * @return if the specified node is a dummy.
     */
    public boolean isNodeDummy(int node) {
        return dummyByNode.get(node);
    }

    /**
     * Returns a collection of all nodes on the specified level.
     * 
     * @param level the level whose nodes are returned.
     * @return a collection of all nodes on the specified level.
     */
    public Collection<Integer> getLevel(int level) {
        if (level < 0 || level >= levelCount) {
            throw new IllegalArgumentException("Illegal level number");
        }

        return Collections.unmodifiableCollection(levels[level]);
    }

    /**
     * Returns the number of nodes on the specified level.
     * 
     * @param level the level.
     * @return the number of nodes on the specified level.
     */
    public int getLevelSize(int level) {
        if (level < 0 || level >= levelCount) {
            throw new IllegalArgumentException("Illegal level number");
        }

        return levels[level].size();
    }
    
    /**
     * Assigns the nodes random positions on their levels. For each level,
     * the order of the nodes is determined by an uniformly drawn permutation.
     * 
     * @param random the source of randomness.
     */
    public void assignPositions(Random random) {
        posByNode = new int[nodeCount];
        for (List<Integer> level : levels) {
            Collections.shuffle(level, random);
            for (int pos = 0; pos < level.size(); pos++) {
                posByNode[level.get(pos)] = pos;
            }
        }
    }

    /**
     * Makes this graph proper leveled by placing new vertices on previously
     * long edges. The new vertices are marked as dummies.
     */
    public void createDummies() {
        int nodeOffset = nodeCount;
        
        for (int source = 0; source < nodeCount; source++) {
            int sourceLevel = levelByNode.get(source);
            for (int target : getNeighbors(source)) {
                int targetLevel = levelByNode.get(target);
                if (targetLevel - sourceLevel > 1) {
                    int dummyCount = targetLevel - sourceLevel - 1;
                    Collection<Integer> sourceNeighbors = neighbors.get(source);
                    sourceNeighbors.remove(target);
                    sourceNeighbors.add(nodeOffset);
                    Collection<Integer> targetNeighbors = neighbors.get(target);
                    targetNeighbors.remove(source);
                    targetNeighbors.add(nodeOffset + dummyCount - 1);
                    
                    for (int dummy = 0; dummy < dummyCount; dummy++) {
                        int currentDummy = nodeOffset + dummy;
                        
                        levelByNode.add(sourceLevel + dummy + 1);
                        dummyByNode.add(true);
                        levels[sourceLevel + dummy + 1].add(currentDummy);
                        
                        int prevNode = dummy == 0 ? source : currentDummy - 1;
                        int nextNode = dummy == dummyCount - 1 ? target : currentDummy + 1;
                        neighbors.add(Arrays.asList(prevNode, nextNode));
                    }
                    
                    nodeOffset += dummyCount;
                }
            }
        }
        
        nodeCount = nodeOffset;
    }
    
    /**
     * Returns if the set of levels with nodes placed on them forms an interval.
     * 
     * @return if the set of levels with nodes forms an interval, i.e.,
     * there are no empty levels between levels with nodes.
     */
    public boolean hasConsecutiveLevels() {
        int level = 0;
        while (level < levelCount && levels[level].isEmpty()) {
            level++;
        }
        
        while (level < levelCount && !levels[level].isEmpty()) {
            level++;
        }
        
        while (level < levelCount) {
            if (!levels[level].isEmpty()) {
                return false;
            }
            
            level++;
        }
        
        return true;
    }
    
    public int getMaxLevelCount() {
        return levelCount;
    }
}
