package de.uni_passau.fim.dagmar;

import static de.uni_passau.fim.dagmar.util.CollectionUtil.swap;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Arrays;
import java.util.Random;

import de.uni_passau.fim.dagmar.arguments.DAGmarProgramArguments;
import de.uni_passau.fim.dagmar.arguments.DAGmarProgramArguments.EmbeddingKind;
import de.uni_passau.fim.dagmar.arguments.ProgramArgumentException;
import de.uni_passau.fim.dagmar.util.UnionFind;

/**
 * The graph generator. This class may be used as a stand-alone program with the
 * {@link #main(String[])} method or as a library utilizing the methods
 * {@link #generate(int, int, boolean, Long)} for directed acyclic graphs or
 * {@link #generate(int, int, boolean, EmbeddingKind, int, int, Long)} for
 * (undirected) leveled graphs.
 */
public class DAGmar {
    
    private static final String USAGE = "Usage: java -jar dagmar.jar "
        + "-n <node count> ( -e <edge count> | -d <density> ) [ -c ]"
        + "[ -l <level count>[,<level size>] [ -p | -el | -ed ] ] [ -s <seed> ] "
        + "[ -f <file pattern> ]\n or\n"
        + "java -jar dagmar.jar multi "
        + "-n <node count ranges> -d <density ranges> [ -i <instance ranges> ] [ -c ] "
        + "[ -l <level count/formula>[,<level size/formula>] [ -p | -el | -ed ] ] "
        + "[ -s <seed> ] [ -flat ] "
        + "-f <file prefix> <target directory>\n";

    public static void main(String[] args) throws IOException {
        if (args.length == 0) {
            System.err.println(USAGE);
            return;
        }
        
        if (args[0].equals("multi")) {
            MultiDAGmar.main(Arrays.copyOfRange(args, 1, args.length));
            return;
        }
        
        try {
            DAGmarProgramArguments arguments = new DAGmarProgramArguments(args);
            File file = arguments.getFile();
            
            PrintStream out = System.out;
            if (file != null) {
                try {
                    out = new PrintStream(new FileOutputStream(file));
                } catch (FileNotFoundException e) {
                    System.err.println(e.getMessage());
                    return;
                }
            }
            
            EmbeddingKind embeddingKind = arguments.getEmbeddingKind();
            boolean isEmbedded = embeddingKind == EmbeddingKind.EMBEDDING
                || embeddingKind == EmbeddingKind.DUMMY_EMBEDDING;
            boolean hasDummies = embeddingKind == EmbeddingKind.DUMMY_EMBEDDING;
            
            DAGmar generator = new DAGmar();
            EmbeddedLevelGraph graph = generator.generate(arguments.getNodeCount(),
                    arguments.getEdgeCount(), arguments.isConnected(),
                    embeddingKind, arguments.getLevelCount(),
                    arguments.getLevelWidth(), arguments.getSeed());
            out.print(new GraphMLBuilder(arguments.isLeveled(), isEmbedded,
                    hasDummies, graph, "G"));
            
            if (file != null) {
                out.close();
            }
        } catch (ProgramArgumentException e) {
            System.err.println(e.getMessage());
        }
    }

    /**
     * Generates a simple directed acyclic graph. This method delegates to the
     * generation of a leveled undirected graph on <code>nodeCount</code> levels
     * with exactly one node on each level. Afterwards, the edges are considered
     * as directed from lower to higher levels and the level assignment is
     * finally discarded.
     * 
     * @param nodeCount
     *            the number of nodes of the graph to create.
     * @param edgeCount
     *            the number of edges of the graph to create.
     * @param connected
     *            denotes if the resulting graph has to be connected.
     * @param seed
     *            the seed for the source of randomness.
     * @return a graph uniformly drawn from the set of all (optionally
     *         connected) simple directed acyclic graphs with <code>nodeCount</code>
     *         nodes and <code>edgeCount</code> edges.
     */
    public EmbeddedLevelGraph generate(int nodeCount, int edgeCount, boolean connected,
            Long seed) {
        return generate(nodeCount, edgeCount, connected, EmbeddingKind.NONE,
                nodeCount, 1, seed);
    }

    /**
     * Generates an undirected simple leveled graph. The nodes are randomly
     * assigned to at most <code>levelCount</code> levels where each level must
     * contain at most <code>levelWidth</code> nodes.
     * 
     * @param nodeCount
     *            the number of nodes of the graph to create.
     * @param edgeCount
     *            the number of edges of the graph to create.
     * @param connected
     *            denotes if the resulting graph has to be connected.
     * @param levelCount
     *            the maximum number of levels the nodes are assigned to.
     * @param levelWidth
     *            the maximum number of nodes per level.
     * @param seed
     *            the seed for the source of randomness. If it is
     *            <code>null</code>, a default source of randomness is created.
     * @return a graph uniformly drawn from the set of all (optionally
     *         connected) simple undirected leveled graphs with
     *         <code>edgeCount</code> edges and <code>nodeCount</code> nodes
     *         randomly placed on at most <code>levelCount</code> levels.
     */
    public EmbeddedLevelGraph generate(int nodeCount, int edgeCount,
            boolean connected, EmbeddingKind embeddingKind, int levelCount,
            int levelWidth, Long seed) {

        // The number edges of the graph to create must not be greater than
        // the maximum number of potential edges for the specified leveling
        // parameters. 
        int maxEdges = countMaxEdges(nodeCount, levelCount, levelWidth,
            embeddingKind == EmbeddingKind.PROPER);
        if (edgeCount > maxEdges) {
            throw new IllegalArgumentException("more edges ("
                + edgeCount + ") as possible (" + maxEdges
                + ") for the specified level parameters");
        }
        
        // Create the source of randomness with the specified seed or a default
        // one if no seed is specified.
        Random random = (seed == null) ? new Random() : new Random(seed);

        // The graph to return.
        EmbeddedLevelGraph graph = new EmbeddedLevelGraph(nodeCount, levelCount);
        
        // Generate the leveling.
        boolean proper = embeddingKind == EmbeddingKind.PROPER;
        boolean success = new LevelingGenerator(random).generateLeveling(graph,
                edgeCount, levelWidth, proper, connected && proper);
        if (!success) {
            throw new AssertionError();
        }
        
        // source[i] is the first node of the i-th potential edge.
        // target[i] is the second node of the i-th potential edge.
        // Edges are considered undirected.
        int source[] = new int[maxEdges];
        int target[] = new int[maxEdges];
        
        // The number of potential edges of the sampled leveling.
        int size = 0;
        
        // Each pair of nodes i and j is tested if the undirected edge
        // {i, j} is compatible (e.g., no intra level edge) with the requested 
        // kind of leveling. If this is the case, it is added to the set
        // of potential edges.
        for (int i = 0; i < nodeCount; i++) {
            int iLevel = graph.getNodeLevel(i);

            for (int j = i + 1; j < nodeCount; j++) {
                int jLevel = graph.getNodeLevel(j);
                
                if ((embeddingKind == EmbeddingKind.PROPER
                            && Math.abs(iLevel - jLevel) == 1)
                        || (embeddingKind != EmbeddingKind.PROPER
                            && iLevel != jLevel)) {
                    source[size] = i;
                    target[size] = j;
                    size++;
                }
            }
        }

        
        // Draw an <code>edgeCount</code>-element subset of the potential edges
        // until it corresponds to a connected graph (or connectedness is not
        // required).
        UnionFind unionFind = new UnionFind(nodeCount);
        do {
            unionFind.reset();
            for (int i = 0; i < edgeCount; i++) {
                int r = random.nextInt(size - i) + i;
                swap(source, i, r);
                swap(target, i, r);
                unionFind.union(source[i], target[i]);
            }
        } while (connected && !unionFind.isConnected());
        
        // Add the drawn edges to the graph to return.
        for (int i = 0; i < edgeCount; i++) {
            graph.addEdge(source[i], target[i]);
        }
        
        if (embeddingKind == EmbeddingKind.DUMMY_EMBEDDING) {
            // Make all edges proper by introducing dummy vertices. 
            graph.createDummies();
        }
        
        if (embeddingKind == EmbeddingKind.EMBEDDING
                || embeddingKind == EmbeddingKind.DUMMY_EMBEDDING) {
            // Draw random permutations defining the order of the nodes on
            // each level.
            graph.assignPositions(random);
        }

        return graph;
    }

    /**
     * Returns the maximum number of potential edges resulting from any
     * level assignment of <code>nodeCount</code> nodes to at most
     * <code>levelCount</code> levels where each level contains at most
     * <code>levelWidth</code> nodes.
     * 
     * @param nodeCount the number of nodes.
     * @param levelCount the maximum number of levels. 
     * @param levelWidth the maximum number of nodes per level.
     * @param isProper denotes if only proper edges are allowed.
     * @return
     */
    public static int countMaxEdges(int nodeCount, int levelCount,
            int levelWidth, boolean isProper) {
        int maxEdgeCount = 0;
        if (isProper) {
            int maxLevelWidth = Math.min(levelWidth, (int) Math.ceil(nodeCount / 2.0));
            int minorWidth = nodeCount % maxLevelWidth;
            int majorWidth = maxLevelWidth;
            int majorCount = nodeCount / maxLevelWidth;
            maxEdgeCount += (majorCount - 1) * majorWidth * majorWidth;
            maxEdgeCount += minorWidth * majorWidth;
        } else {
            int minorWidth = nodeCount / levelCount;
            int majorWidth = minorWidth + 1;
            int majorCount = nodeCount % levelCount;
            int minorCount = levelCount - majorCount;
            maxEdgeCount += minorCount * (minorCount - 1) / 2 * minorWidth * minorWidth;
            maxEdgeCount += majorCount * (majorCount - 1) / 2 * majorWidth * majorWidth;
            maxEdgeCount += minorCount * majorCount * minorWidth * majorWidth;            
        }
        
        return maxEdgeCount;
    }
}
